//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;

using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents the base class from which all implementations of constructor parameter attributes must derive to describe constructor parameters.
	/// </summary>
	[AttributeUsage(AttributeTargets.Class, AllowMultiple=true, Inherited=false)]
	public abstract class ConstructorParameterAttribute : Attribute
	{
		private readonly int order;
		private readonly Type parameterType;

		/// <summary>
		/// Initialize a new instance of the <see cref="ConstructorParameterAttribute"/> class with the order and parameter type.
		/// </summary>
		/// <param name="order">The ordinal position of the constructor parameter.</param>
		/// <param name="parameterType">The parameter type.</param>
		protected ConstructorParameterAttribute(int order, Type parameterType)
		{
			this.order = order;
			this.parameterType = parameterType;
		}

		/// <summary>
		/// Gets the ordinal position of the constructor parameter.
		/// </summary>
		/// <value>
		/// The ordinal position of the constructor parameter.
		/// </value>
		public int Order
		{
			get { return order; }		
		}

		/// <summary>
		/// Gets the parameter type for the constructor.
		/// </summary>
		/// <value>
		/// The parameter type for the constructor.
		/// </value>
		public Type ParameterType
		{
			get { return parameterType; }
		}

		/// <summary>
		/// When overridden in a class, gets the <see cref="ParameterStrategy"/> used to create the parameter value.
		/// </summary>
		/// <param name="objectConfiguration">The object configuration used to retrieve the value.</param>
		/// <returns>A <see cref="ParameterStrategy"/> object.</returns>
        protected internal abstract IObjectFactory<TObjectToCreate, TObjectConfiguration> GetParameterFactory<TObjectToCreate, TObjectConfiguration>(TObjectConfiguration objectConfiguration);		
	}
}
